
/************************************************************************
 *
 * \file: SignalHandler.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto - Demo application
 *
 * \author: J. Harder / ADITG/SW1 / jharder@de.adit-jv.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <signal.h>
#include <adit_logging.h>
#include <sys/signalfd.h>
#include <signal.h>
// memset
#include <string.h>
#include <errno.h>
#include <assert.h>
#include <sys/prctl.h>
#include "SignalHandler.h"

LOG_IMPORT_CONTEXT(tbdcl)

namespace adit { namespace bdcl {

SignalHandler::SignalHandler()
{
    mSignaled = false;
    mSignalFd = -1;

    sigset_t mask;

    //  initializes the signal mask to empty
    sigemptyset(&mask);
    // add respectively signal to signal mask
    sigaddset(&mask, SIGTERM);
    sigaddset(&mask, SIGUSR1);
    sigaddset(&mask, SIGINT);
    sigaddset(&mask, SIGQUIT);
    sigaddset(&mask, SIGHUP);

    // use sigprocmask to fetch the signal mask
    if (sigprocmask(SIG_BLOCK, &mask, NULL) == -1) {
    	fprintf(stderr, " Could not set procmask \n");
    	assert(&mask);
    }

    // create new file descriptor and associates the signal mask with that descriptor.
    mSignalFd = signalfd(-1, &mask, 0);
    if (mSignalFd < 0) {
    	fprintf(stderr, "Wrong parameter to add event fd=%d \n", mSignalFd);
    	assert(&mSignalFd);
    }

    /* create and open epoll descriptor */
    mEpollFd = epoll_create1(0);
    if (0 > mEpollFd) {
    	fprintf(stderr, "epoll_create1 error: %d \n", mEpollFd);
    	assert(&mEpollFd);
    }

    mThreadId = -1;
    mRunning = true;

    // create thread to poll signals targeted at the caller
    int32_t err = pthread_create(&mThreadId, nullptr, &signalHandlerThread, this);
    if (0 != err) {
    	fprintf(stderr, "create signalHandlerThread failed err=%d, errno=%d \n", err, errno);
    	assert(&mThreadId);
    }
}

SignalHandler::~SignalHandler()
{
    mRunning = false;

    // close epoll descriptor
    if (mEpollFd >= 0) {
        close(mEpollFd);
        mEpollFd = -1;
    }

    if (mThreadId > 0) {
        pthread_join(mThreadId, nullptr);
        mThreadId = -1;
    }

    // close signal file descriptor
    if (mSignalFd >= 0) {
        close(mSignalFd);
        mSignalFd = -1;
    }
}

void SignalHandler::registerForSIGINT(IStoppable* inStoppable)
{
    // TODO mutex lock
    stoppables.push_back(inStoppable);
}

void SignalHandler::unregister(void* inPointer)
{
    // TODO mutex lock
    for (auto iter = stoppables.begin(); iter != stoppables.end(); iter++)
    {
        if (*iter == inPointer)
        {
            stoppables.erase(iter);
            break;
        }
    }
}

int32_t SignalHandler::signalHandlerCb(void* context)
{
    int32_t res = 0;

    if (nullptr != context) {

        if (!instance().mSignaled) {
            instance().mSignaled= true;

            fprintf(stderr, "signal received \n");
            // TODO mutex lock
            for (auto item : instance().stoppables)
            {
                item->requestStop();
            }
        } else {
            fprintf(stderr, "signalhandler already signaled \n");
        }
    } else {
        fprintf(stderr, "could not cast input pointer \n");
        res = -1;
    }

    return res;
}

int32_t SignalHandler::epollRegisterSignalFd(int32_t epollFd)
{
    struct epoll_event event;

    event.data.ptr = (void*)SignalHandler::signalHandlerCb;
    event.events = EPOLLIN;

    return epoll_ctl(epollFd, EPOLL_CTL_ADD, mSignalFd, &event);
}

int32_t SignalHandler::epollWaitSignalFd(int32_t epollFd, struct epoll_event* epollEvents)
{
    int32_t n = 0;
    int32_t res = -1;
    int32_t i = 0;

    if (!epollEvents) {
        fprintf(stderr, "No memory available for events \n");
        return -1;
    }

    n = epoll_wait(epollFd, epollEvents, 10, 500);
    if (n < 0) {
        fprintf(stderr, "epoll_wait error: n=%d, %s \n", n, strerror(errno));

        if (errno == EINTR) {
            /* Only exit if the daemon has received QUIT/INT/TERM */
            return 0;
        }
        return -1;
    } else if (n == 0) {
//        fprintf(stderr, "epoll_wait timed out: n=%d \n", n);
        res = 0;
    } else {
        for (i = 0 ; i < n ; i++)
        {
            int32_t (*callback)(void *context) = (int32_t (*)(void*))epollEvents[i].data.ptr;
            if (!callback)
            {
                fprintf(stderr, "Callback not found, exiting \n");
                break;
            }

            if (!(epollEvents[i].events & (EPOLLIN | EPOLLET)))
            {
                fprintf(stderr, "Error while polling. Event received: 0x%X \n", epollEvents[i].events);

                /* We only support one event producer.
                 * Error means that this producer died.
                 */
                break;
            }

            struct signalfd_siginfo info;
            ssize_t bytes = read(mSignalFd, &info, sizeof(info));
            if (bytes > 0) {
                uint32_t sig = info.ssi_signo;
                switch(sig)
                {
                    case SIGTERM:
                        printf("\n Got SIGUSR1 \n");
                        break;
                    case SIGUSR1:
                        printf("\n Got SIGUSR1 \n");
                        break;
                    case SIGINT:
                        printf("\n Got SIGINT \n");
                        break;
                    case SIGQUIT:
                        printf("\n Got SIGQUIT \n");
                        break;
                    case SIGHUP:
                        printf("\n Got SIGHUP \n");
                        break;
                    default:
                        printf("\n Got unknown signal %u \n", sig);
                        break;
                }
            } else {
                printf("read(%d) returned %zd) \n", mSignalFd, bytes);
            }


            if (callback(this) < 0)
            {
                fprintf(stderr, "Error while calling the callback, exiting \n");
                break;
            }
            else
            {
                fprintf(stderr, "Callback return successfully \n");
                res = 0;
            }
        }
    }

    return res;
}

void* SignalHandler::signalHandlerThread(void* context)
{
    int32_t res = 0;

    auto me = static_cast<SignalHandler*>(context);
    if (me == nullptr) {
        fprintf(stderr, "could not cast input pointer \n");
        return nullptr;
    }

    // set thread name
    prctl(PR_SET_NAME, "bdcl-demo-sighndl", 0, 0, 0);

    if (0 != me->epollRegisterSignalFd(me->mEpollFd)) {
        fprintf(stderr, "could not register epoll event for fd=%d \n", me->mEpollFd);
        return nullptr;
    } else {
        fprintf(stderr, "registered epoll event \n");
    }

    struct epoll_event* epollEvents = new epoll_event[10];
    memset(epollEvents, 0, 10 * sizeof(epoll_event));

    while ((me->mRunning) && (0 == res))
    {
        res = me->epollWaitSignalFd(me->mEpollFd, epollEvents);
        if (0 > res) {
            fprintf(stderr, "epollWaitSignalFd error: %d \n", res);
        }
        memset(epollEvents, 0, 10 * sizeof(epoll_event));
    }

    delete[] epollEvents;
    epollEvents = nullptr;

    return nullptr;
}
} } // namespace adit { namespace bdcl {
